/*
 * Decompiled with CFR 0.152.
 */
package org.python.google.common.net;

import com.google.common.annotations.Beta;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Locale;
import javax.annotation.Nullable;
import org.python.google.common.base.Preconditions;
import org.python.google.common.io.ByteStreams;
import org.python.google.common.primitives.Ints;
import org.python.google.common.primitives.Longs;

@Beta
public final class InetAddresses {
    private static final int IPV4_PART_COUNT = 4;
    private static final int IPV6_PART_COUNT = 8;
    private static final Inet4Address LOOPBACK4 = (Inet4Address)InetAddresses.forString("127.0.0.1");
    private static final Inet4Address ANY4 = (Inet4Address)InetAddresses.forString("0.0.0.0");

    private InetAddresses() {
    }

    private static Inet4Address getInet4Address(byte[] byArray) {
        Preconditions.checkArgument(byArray.length == 4, "Byte array has invalid length for an IPv4 address: %s != 4.", byArray.length);
        try {
            InetAddress inetAddress = InetAddress.getByAddress(byArray);
            if (!(inetAddress instanceof Inet4Address)) {
                throw new UnknownHostException(String.format("'%s' is not an IPv4 address.", inetAddress.getHostAddress()));
            }
            return (Inet4Address)inetAddress;
        }
        catch (UnknownHostException unknownHostException) {
            throw new IllegalArgumentException(String.format("Host address '%s' is not a valid IPv4 address.", Arrays.toString(byArray)), unknownHostException);
        }
    }

    public static InetAddress forString(String string) {
        byte[] byArray = InetAddresses.textToNumericFormatV4(string);
        if (byArray == null) {
            byArray = InetAddresses.textToNumericFormatV6(string);
        }
        if (byArray == null) {
            throw new IllegalArgumentException(String.format("'%s' is not an IP string literal.", string));
        }
        try {
            return InetAddress.getByAddress(byArray);
        }
        catch (UnknownHostException unknownHostException) {
            throw new IllegalArgumentException(String.format("'%s' is extremely broken.", string), unknownHostException);
        }
    }

    public static boolean isInetAddress(String string) {
        try {
            InetAddresses.forString(string);
            return true;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            return false;
        }
    }

    private static byte[] textToNumericFormatV4(String string) {
        boolean bl = false;
        if (string.toUpperCase(Locale.US).startsWith("::FFFF:")) {
            string = string.substring(7);
        } else if (string.startsWith("::")) {
            string = string.substring(2);
            bl = true;
        }
        String[] stringArray = string.split("\\.");
        if (stringArray.length != 4) {
            return null;
        }
        try {
            byte[] byArray = new byte[4];
            for (int i = 0; i < byArray.length; ++i) {
                int n = Integer.parseInt(stringArray[i]);
                if (n < 0 || n > 255) {
                    return null;
                }
                if (stringArray[i].startsWith("0") && stringArray[i].length() != 1) {
                    return null;
                }
                byArray[i] = (byte)n;
            }
            if (bl) {
                byte[] byArray2 = new byte[16];
                System.arraycopy(byArray, 0, byArray2, 12, 4);
                return byArray2;
            }
            return byArray;
        }
        catch (NumberFormatException numberFormatException) {
            return null;
        }
    }

    private static byte[] textToNumericFormatV6(String string) {
        if (!string.contains(":")) {
            return null;
        }
        if (string.contains(":::")) {
            return null;
        }
        if (string.contains(".") && (string = InetAddresses.convertDottedQuadToHex(string)) == null) {
            return null;
        }
        string = InetAddresses.padIpString(string);
        try {
            String[] stringArray = string.split(":", 8);
            if (stringArray.length != 8) {
                return null;
            }
            byte[] byArray = new byte[16];
            for (int i = 0; i < 8; ++i) {
                int n = stringArray[i].equals("") ? 0 : Integer.parseInt(stringArray[i], 16);
                byArray[2 * i] = (byte)((n & 0xFF00) >>> 8);
                byArray[2 * i + 1] = (byte)(n & 0xFF);
            }
            return byArray;
        }
        catch (NumberFormatException numberFormatException) {
            return null;
        }
    }

    private static String padIpString(String string) {
        if (string.contains("::")) {
            int n = InetAddresses.numberOfColons(string);
            StringBuilder stringBuilder = new StringBuilder("::");
            int n2 = 0;
            while (n2 + n < 7) {
                stringBuilder.append(":");
                ++n2;
            }
            string = string.replace("::", stringBuilder);
        }
        return string;
    }

    private static int numberOfColons(String string) {
        int n = 0;
        for (int i = 0; i < string.length(); ++i) {
            if (string.charAt(i) != ':') continue;
            ++n;
        }
        return n;
    }

    private static String convertDottedQuadToHex(String string) {
        int n = string.lastIndexOf(58);
        String string2 = string.substring(0, n + 1);
        String string3 = string.substring(n + 1);
        byte[] byArray = InetAddresses.textToNumericFormatV4(string3);
        if (byArray == null) {
            return null;
        }
        String string4 = Integer.toHexString((byArray[0] & 0xFF) << 8 | byArray[1] & 0xFF);
        String string5 = Integer.toHexString((byArray[2] & 0xFF) << 8 | byArray[3] & 0xFF);
        return string2 + string4 + ":" + string5;
    }

    public static String toUriString(InetAddress inetAddress) {
        if (inetAddress instanceof Inet6Address) {
            return "[" + inetAddress.getHostAddress() + "]";
        }
        return inetAddress.getHostAddress();
    }

    public static InetAddress forUriString(String string) {
        Preconditions.checkNotNull(string);
        Preconditions.checkArgument(string.length() > 0, "host string is empty");
        InetAddress inetAddress = null;
        try {
            inetAddress = InetAddresses.forString(string);
            if (inetAddress instanceof Inet4Address) {
                return inetAddress;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        if (!string.startsWith("[") || !string.endsWith("]")) {
            throw new IllegalArgumentException("Not a valid address: \"" + string + '\"');
        }
        inetAddress = InetAddresses.forString(string.substring(1, string.length() - 1));
        if (inetAddress instanceof Inet6Address) {
            return inetAddress;
        }
        throw new IllegalArgumentException("Not a valid address: \"" + string + '\"');
    }

    public static boolean isUriInetAddress(String string) {
        try {
            InetAddresses.forUriString(string);
            return true;
        }
        catch (IllegalArgumentException illegalArgumentException) {
            return false;
        }
    }

    public static boolean isCompatIPv4Address(Inet6Address inet6Address) {
        if (!inet6Address.isIPv4CompatibleAddress()) {
            return false;
        }
        byte[] byArray = inet6Address.getAddress();
        return byArray[12] != 0 || byArray[13] != 0 || byArray[14] != 0 || byArray[15] != 0 && byArray[15] != 1;
    }

    public static Inet4Address getCompatIPv4Address(Inet6Address inet6Address) {
        Preconditions.checkArgument(InetAddresses.isCompatIPv4Address(inet6Address), "Address '%s' is not IPv4-compatible.", inet6Address.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(inet6Address.getAddress(), 12, 16));
    }

    public static boolean is6to4Address(Inet6Address inet6Address) {
        byte[] byArray = inet6Address.getAddress();
        return byArray[0] == 32 && byArray[1] == 2;
    }

    public static Inet4Address get6to4IPv4Address(Inet6Address inet6Address) {
        Preconditions.checkArgument(InetAddresses.is6to4Address(inet6Address), "Address '%s' is not a 6to4 address.", inet6Address.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(inet6Address.getAddress(), 2, 6));
    }

    public static boolean isTeredoAddress(Inet6Address inet6Address) {
        byte[] byArray = inet6Address.getAddress();
        return byArray[0] == 32 && byArray[1] == 1 && byArray[2] == 0 && byArray[3] == 0;
    }

    public static TeredoInfo getTeredoInfo(Inet6Address inet6Address) {
        Preconditions.checkArgument(InetAddresses.isTeredoAddress(inet6Address), "Address '%s' is not a Teredo address.", inet6Address.getHostAddress());
        byte[] byArray = inet6Address.getAddress();
        Inet4Address inet4Address = InetAddresses.getInet4Address(InetAddresses.copyOfRange(byArray, 4, 8));
        int n = ByteStreams.newDataInput(byArray, 8).readShort() & 0xFFFF;
        int n2 = ~ByteStreams.newDataInput(byArray, 10).readShort() & 0xFFFF;
        byte[] byArray2 = InetAddresses.copyOfRange(byArray, 12, 16);
        for (int i = 0; i < byArray2.length; ++i) {
            byArray2[i] = ~byArray2[i];
        }
        Inet4Address inet4Address2 = InetAddresses.getInet4Address(byArray2);
        return new TeredoInfo(inet4Address, inet4Address2, n2, n);
    }

    public static boolean isIsatapAddress(Inet6Address inet6Address) {
        if (InetAddresses.isTeredoAddress(inet6Address)) {
            return false;
        }
        byte[] byArray = inet6Address.getAddress();
        if ((byArray[8] | 3) != 3) {
            return false;
        }
        return byArray[9] == 0 && byArray[10] == 94 && byArray[11] == -2;
    }

    public static Inet4Address getIsatapIPv4Address(Inet6Address inet6Address) {
        Preconditions.checkArgument(InetAddresses.isIsatapAddress(inet6Address), "Address '%s' is not an ISATAP address.", inet6Address.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(inet6Address.getAddress(), 12, 16));
    }

    public static boolean hasEmbeddedIPv4ClientAddress(Inet6Address inet6Address) {
        return InetAddresses.isCompatIPv4Address(inet6Address) || InetAddresses.is6to4Address(inet6Address) || InetAddresses.isTeredoAddress(inet6Address);
    }

    public static Inet4Address getEmbeddedIPv4ClientAddress(Inet6Address inet6Address) {
        if (InetAddresses.isCompatIPv4Address(inet6Address)) {
            return InetAddresses.getCompatIPv4Address(inet6Address);
        }
        if (InetAddresses.is6to4Address(inet6Address)) {
            return InetAddresses.get6to4IPv4Address(inet6Address);
        }
        if (InetAddresses.isTeredoAddress(inet6Address)) {
            return InetAddresses.getTeredoInfo(inet6Address).getClient();
        }
        throw new IllegalArgumentException(String.format("'%s' has no embedded IPv4 address.", inet6Address.getHostAddress()));
    }

    public static Inet4Address getCoercedIPv4Address(InetAddress inetAddress) {
        if (inetAddress instanceof Inet4Address) {
            return (Inet4Address)inetAddress;
        }
        byte[] byArray = inetAddress.getAddress();
        boolean bl = true;
        for (int i = 0; i < 15; ++i) {
            if (byArray[i] == 0) continue;
            bl = false;
            break;
        }
        if (bl && byArray[15] == 1) {
            return LOOPBACK4;
        }
        if (bl && byArray[15] == 0) {
            return ANY4;
        }
        Inet6Address inet6Address = (Inet6Address)inetAddress;
        long l = 0L;
        l = InetAddresses.hasEmbeddedIPv4ClientAddress(inet6Address) ? (long)InetAddresses.getEmbeddedIPv4ClientAddress(inet6Address).hashCode() : ByteBuffer.wrap(inet6Address.getAddress(), 0, 8).getLong();
        int n = Longs.hashCode(l);
        if ((n |= 0xE0000000) == -1) {
            n = -2;
        }
        return InetAddresses.getInet4Address(Ints.toByteArray(n));
    }

    public static int coerceToInteger(InetAddress inetAddress) {
        return ByteStreams.newDataInput(InetAddresses.getCoercedIPv4Address(inetAddress).getAddress()).readInt();
    }

    public static Inet4Address fromInteger(int n) {
        return InetAddresses.getInet4Address(Ints.toByteArray(n));
    }

    public static InetAddress fromLittleEndianByteArray(byte[] byArray) throws UnknownHostException {
        byte[] byArray2 = new byte[byArray.length];
        for (int i = 0; i < byArray.length; ++i) {
            byArray2[i] = byArray[byArray.length - i - 1];
        }
        return InetAddress.getByAddress(byArray2);
    }

    private static byte[] copyOfRange(byte[] byArray, int n, int n2) {
        Preconditions.checkNotNull(byArray);
        int n3 = Math.min(n2, byArray.length);
        byte[] byArray2 = new byte[n2 - n];
        System.arraycopy(byArray, n, byArray2, 0, n3 - n);
        return byArray2;
    }

    @Beta
    public static final class TeredoInfo {
        private final Inet4Address server;
        private final Inet4Address client;
        private final int port;
        private final int flags;

        public TeredoInfo(@Nullable Inet4Address inet4Address, @Nullable Inet4Address inet4Address2, int n, int n2) {
            Preconditions.checkArgument(n >= 0 && n <= 65535, "port '%d' is out of range (0 <= port <= 0xffff)", n);
            Preconditions.checkArgument(n2 >= 0 && n2 <= 65535, "flags '%d' is out of range (0 <= flags <= 0xffff)", n2);
            this.server = inet4Address != null ? inet4Address : ANY4;
            this.client = inet4Address2 != null ? inet4Address2 : ANY4;
            this.port = n;
            this.flags = n2;
        }

        public Inet4Address getServer() {
            return this.server;
        }

        public Inet4Address getClient() {
            return this.client;
        }

        public int getPort() {
            return this.port;
        }

        public int getFlags() {
            return this.flags;
        }
    }
}

